// Lawyer Consultation Website - Version 4 (HTML/CSS/JS)

const areas = [
  { 
    id: 'family', 
    name: 'Семейное право', 
    icon: 'icons/area-family.svg',
    description: 'Разводы, алименты, опека'
  },
  { 
    id: 'labor', 
    name: 'Трудовое право', 
    icon: 'icons/area-labor.svg',
    description: 'Трудовые споры, увольнения'
  },
  { 
    id: 'housing', 
    name: 'Жилищное право', 
    icon: 'icons/area-housing.svg',
    description: 'Сделки с недвижимостью'
  },
  { 
    id: 'criminal', 
    name: 'Уголовное право', 
    icon: 'icons/area-criminal.svg',
    description: 'Защита по уголовным делам'
  }
]

const lawyers = [
  { name: 'Анна Соколова', experience: '15 лет', rating: 5, specialty: 'Семейное, Жилищное право' },
  { name: 'Михаил Петров', experience: '12 лет', rating: 5, specialty: 'Трудовое, Корпоративное право' },
  { name: 'Елена Волкова', experience: '18 лет', rating: 5, specialty: 'Уголовное, Административное право' },
  { name: 'Дмитрий Иванов', experience: '10 лет', rating: 5, specialty: 'Жилищное, Семейное право' }
]

const features = [
  { icon: 'icons/stat-status.svg', title: 'Защита прав', text: 'Максимальная защита интересов' },
  { icon: 'icons/stat-lawyer.svg', title: 'Опытные юристы', text: 'Специалисты с опытом' },
  { icon: 'icons/form-calendar.svg', title: 'Быстрая помощь', text: 'Консультации в удобное время' },
  { icon: 'icons/stat-consultations.svg', title: 'Высокий рейтинг', text: '98% довольных клиентов' }
]

const guarantees = [
  { icon: 'icons/stat-status.svg', title: 'Конфиденциальность', text: 'Защита персональных данных' },
  { icon: 'icons/stat-lawyer.svg', title: 'Профессионализм', text: 'Опытные специалисты' },
  { icon: 'icons/form-calendar.svg', title: 'Оперативность', text: 'Быстрые ответы' },
  { icon: 'icons/stat-consultations.svg', title: 'Результативность', text: '95% успешных дел' }
]

function initPage() {
  const currentPage = window.location.pathname.includes('new.html') ? 'new' : 'index'
  
  if (currentPage === 'new') {
    initNewPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadConsultations()
  updateStats()
  renderFeatures()
  
  setInterval(() => {
    loadConsultations()
    updateStats()
  }, 500)
}

function loadConsultations() {
  const saved = localStorage.getItem('lawConsultations')
  let consultations = []
  
  if (saved) {
    try {
      consultations = JSON.parse(saved)
    } catch (e) {
      consultations = []
    }
  } else {
    consultations = [
      {
        id: 1,
        topic: 'Семейное право',
        lawyer: 'Анна Соколова',
        status: 'Завершено',
        review: 'Отличная консультация, все разъяснили подробно'
      },
      {
        id: 2,
        topic: 'Трудовое право',
        lawyer: 'Михаил Петров',
        status: 'В процессе',
        review: null
      },
      {
        id: 3,
        topic: 'Жилищное право',
        lawyer: 'Елена Волкова',
        status: 'Завершено',
        review: 'Профессионал своего дела, рекомендую!'
      }
    ]
    localStorage.setItem('lawConsultations', JSON.stringify(consultations))
  }
  
  renderConsultations(consultations)
  return consultations
}

function getStatusColor(status) {
  return status === 'Завершено' ? '#10B981' : '#F59E0B'
}

function renderConsultations(consultations) {
  const consultationsVertical = document.getElementById('consultationsVertical')
  if (!consultationsVertical) return
  
  if (consultations.length === 0) {
    consultationsVertical.innerHTML = `
      <div class="empty-state">
        <img src="icons/form-scale.svg" alt="" class="empty-icon" width="64" height="64" />
        <h3 class="empty-title">Нет обращений</h3>
        <p class="empty-text">Запишитесь на консультацию</p>
      </div>
    `
  } else {
    consultationsVertical.innerHTML = consultations.map(consultation => {
      const statusColor = getStatusColor(consultation.status)
      
      return `
        <div class="consultation-item">
          <div class="consultation-item-left">
            <div class="consultation-item-icon">
              <img src="icons/form-scale.svg" alt="" width="24" height="24" />
            </div>
            <div class="consultation-item-content">
              <div class="consultation-item-header">
                <h3 class="consultation-item-topic">${consultation.topic}</h3>
                <div class="consultation-item-status" style="background: ${statusColor}20; color: ${statusColor};">
                  ${consultation.status}
                </div>
              </div>
              <div class="consultation-item-details">
                <div class="consultation-item-detail">
                  <img src="icons/stat-lawyer.svg" alt="" width="16" height="16" />
                  <span>${consultation.lawyer}</span>
                </div>
                ${consultation.review ? `
                  <div class="consultation-item-review">
                    <img src="icons/form-scale.svg" alt="" width="16" height="16" />
                    <span>${consultation.review}</span>
                  </div>
                ` : ''}
              </div>
            </div>
          </div>
        </div>
      `
    }).join('')
  }
}

function updateStats() {
  const consultations = loadConsultations()
  const stats = [
    { 
      value: consultations.length, 
      label: 'Обращений'
    },
    { 
      value: consultations.filter(c => c.status === 'Завершено').length, 
      label: 'Завершено'
    },
    { 
      value: new Set(consultations.map(c => c.lawyer)).size, 
      label: 'Юристов'
    }
  ]
  
  const statsMinimal = document.getElementById('statsMinimal')
  if (statsMinimal) {
    statsMinimal.innerHTML = stats.map(stat => `
      <div class="stat-minimal-item">
        <div class="stat-minimal-value">${stat.value}</div>
        <div class="stat-minimal-label">${stat.label}</div>
      </div>
    `).join('')
  }
}

function renderFeatures() {
  const featuresGrid = document.getElementById('featuresGrid')
  if (featuresGrid) {
    featuresGrid.innerHTML = features.map(feature => `
      <div class="feature-item">
        <div class="feature-icon">
          <img src="${feature.icon}" alt="" width="24" height="24" />
        </div>
        <div class="feature-content">
          <h3 class="feature-title">${feature.title}</h3>
          <p class="feature-text">${feature.text}</p>
        </div>
      </div>
    `).join('')
  }
}

function initNewPage() {
  const consultationForm = document.getElementById('consultationForm')
  const areasList = document.getElementById('areasList')
  const dateInput = document.getElementById('date')
  const successMessage = document.getElementById('successMessage')
  const lawyersList = document.getElementById('lawyersList')
  const guaranteesGrid = document.getElementById('guaranteesGrid')

  let selectedArea = null

  if (areasList) {
    areasList.innerHTML = areas.map(area => `
      <button type="button" class="area-item" data-id="${area.id}">
        <div class="area-item-icon">
          <img src="${area.icon}" alt="${area.name}" width="24" height="24" />
        </div>
        <div class="area-item-content">
          <h3 class="area-item-name">${area.name}</h3>
          <p class="area-item-desc">${area.description}</p>
        </div>
      </button>
    `).join('')

    areasList.querySelectorAll('.area-item').forEach(button => {
      button.addEventListener('click', function() {
        areasList.querySelectorAll('.area-item').forEach(btn => {
          btn.classList.remove('selected')
          const check = btn.querySelector('.area-item-check')
          if (check) check.remove()
        })
        this.classList.add('selected')
        selectedArea = areas.find(a => a.id === this.dataset.id)
        
        const check = document.createElement('div')
        check.className = 'area-item-check'
        check.innerHTML = `<img src="icons/btn-check.svg" alt="" width="18" height="18" />`
        this.appendChild(check)
      })
    })
  }

  if (lawyersList) {
    lawyersList.innerHTML = lawyers.map(lawyer => {
      const stars = Array.from({ length: 5 }, () => 
        `<img src="icons/stat-status.svg" alt="star" width="14" height="14" style="filter: drop-shadow(0 0 4px rgba(245, 158, 11, 0.5));" />`
      ).join('')
      
      return `
        <div class="lawyer-item">
          <div class="lawyer-item-left">
            <div class="lawyer-item-icon">
              <img src="icons/stat-lawyer.svg" alt="" width="24" height="24" />
            </div>
            <div class="lawyer-item-content">
              <h3 class="lawyer-item-name">${lawyer.name}</h3>
              <div class="lawyer-item-details">
                <span class="lawyer-item-exp">${lawyer.experience} опыта</span>
                <div class="lawyer-item-rating">
                  ${stars}
                </div>
              </div>
              <p class="lawyer-item-specialty">${lawyer.specialty}</p>
            </div>
          </div>
        </div>
      `
    }).join('')
  }

  if (guaranteesGrid) {
    guaranteesGrid.innerHTML = guarantees.map(guarantee => `
      <div class="guarantee-item">
        <div class="guarantee-icon">
          <img src="${guarantee.icon}" alt="" width="24" height="24" />
        </div>
        <div class="guarantee-content">
          <h3 class="guarantee-title">${guarantee.title}</h3>
          <p class="guarantee-text">${guarantee.text}</p>
        </div>
      </div>
    `).join('')
  }

  if (dateInput) {
    const minDate = new Date().toISOString().split('T')[0]
    const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]
    dateInput.setAttribute('min', minDate)
    dateInput.setAttribute('max', maxDate)
  }

  if (consultationForm) {
    consultationForm.addEventListener('submit', function(e) {
      e.preventDefault()
      if (selectedArea && dateInput.value) {
        const savedConsultations = localStorage.getItem('lawConsultations')
        const consultations = savedConsultations ? JSON.parse(savedConsultations) : []
        
        const randomLawyer = lawyers[Math.floor(Math.random() * lawyers.length)]
        const newConsultation = {
          id: Date.now(),
          topic: selectedArea.name,
          lawyer: randomLawyer.name,
          status: 'В процессе',
          review: null
        }

        consultations.push(newConsultation)
        localStorage.setItem('lawConsultations', JSON.stringify(consultations))
        
        successMessage.style.display = 'flex'
        setTimeout(() => {
          successMessage.style.display = 'none'
          selectedArea = null
          dateInput.value = ''
          areasList.querySelectorAll('.area-item').forEach(btn => {
            btn.classList.remove('selected')
            const check = btn.querySelector('.area-item-check')
            if (check) check.remove()
          })
          window.location.href = 'index.html'
        }, 2000)
      }
    })
  }
}

let currentSlide = 0
let slides = []
let dots = []
let slideInterval

function initSlider() {
  const sliderContainer = document.getElementById('slider')
  if (!sliderContainer) return

  slides = Array.from(sliderContainer.querySelectorAll('.slide'))
  dots = Array.from(sliderContainer.querySelectorAll('.dot'))
  
  const sliderPrev = document.getElementById('sliderPrev')
  const sliderNext = document.getElementById('sliderNext')

  if (sliderPrev) sliderPrev.addEventListener('click', () => changeSlide(currentSlide - 1))
  if (sliderNext) sliderNext.addEventListener('click', () => changeSlide(currentSlide + 1))
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => changeSlide(index))
  })

  startSlider()
}

function startSlider() {
  stopSlider()
  slideInterval = setInterval(() => {
    changeSlide(currentSlide + 1)
  }, 3000)
}

function stopSlider() {
  clearInterval(slideInterval)
}

function changeSlide(n) {
  slides[currentSlide].classList.remove('active')
  dots[currentSlide].classList.remove('active')

  currentSlide = (n + slides.length) % slides.length
  
  slides[currentSlide].classList.add('active')
  dots[currentSlide].classList.add('active')
  
  startSlider()
}

document.addEventListener('DOMContentLoaded', initPage)

